// ============================================================================
// ============================================================================
// ============================================================================
// ==                                                                        ==
// == Name    : TheEmuLib.Emu_Rectangle.fsh                                  ==
// == Type    : Fragment shader                                              ==
// == Version : 1.0.2 (2015/12/16)                                           ==
// == Creator : TheEmu © TheEmu 2017, Some Rights Reserved                   ==                                                   ==
// == Licence : Creative Commons Attribution-ShareAlike 4.0                  ==
// ==           http://creativecommons.org/licences/by-sa/4.0                ==
// ==                                                                        ==
// == Purpose: To modify opacities inside and outside a rectangular area.    ==
// ==                                                                        ==
// == Description:  A source image is manipulated by modifying the opacities ==
// == both inside and outside a rectangular region defined by  the  shader's ==
// == inputs. The opacities are modified by multiplying the  images's  alpha ==
// == channel values by different factors inside and outside the rectangle.  ==
// ==                                                                        ==
// == The rectangle can be optionaly be surrounded, outside or inside,  by a ==
// == border where the opacity is smoothly varied from its value used inside == 
// == the rectangle to that used outside.                                    ==
// ==                                                                        ==
// == A common use is expected to be to discard either the  area  inside  or ==
// == outside the rectangle by specifying an opacity factor of 0.0 for  that ==
// == region and  a factor greater than 0.0 for the other  region,  i.e.  to ==
// == extract a part of an image or to punch a rectangular window in one but ==
// == it is much more versatile than that and can be used,  for example,  to ==  
// == implement vignetting.                                                  ==
// ==                                                                        ==
// == This file is a member of The Emu's shader library.                     ==
// ==                                                                        ==
// == ====================================================================== ==
// ==                                                                        ==
// == Update history:                                                        ==
// ==                                                                        ==
// ==   2016/12/07 - v1.0.0 - Initial version.                               ==
// ==   2016/12/12 - v1.0.1 - Cleaned up a little.                           ==
// ==   2016/12/16 - v1.0.2 - Regularised format of TheEmuLib sources.       ==
// ==                                                                        ==
// ============================================================================
// ============================================================================
// ============================================================================

// ============================================================================
// == Standard shader inputs ==================================================
// ============================================================================

uniform float u_Elapsed;    // The elapsed time in seconds
uniform vec2  u_WindowSize; // Window dimensions in pixels

// The image that is to be manipulated.

uniform sampler2D iChannel0;

// ============================================================================
// == Imports from TheEmuLib ==================================================
// ============================================================================
//
// The GLSL shader language currently provides no mechanism for importing  any
// elements that are defined in other modules, not even C's crude source level
// #include mechanism. In the absence of anything better TheEmuLib handles any
// imports by manualy copying relevent utility code snippets from the  sources
// in the Shader Lib.Inc directory. This is very crude but I have attempted to
// be systematic in the way in which this is presented in the library sources.
//
// = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =

// Macros from TheEmuLib.Emu_Common_Utilities.lib.src

#define EMU_DEFAULT(type,x,default_value) (x==type(0.0)) ? (default_value) : (x)

// = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =

// Functions from TheEmuLib.Emu_Coordinate_Normalisation.lib.src

#define EMU_NORMALISE_TO_WINDOW_1(xy,wsize) ( (xy)/(wsize) )

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

vec2 Emu_Normalise_to_Window ( vec2 xy )
 { return EMU_NORMALISE_TO_WINDOW_1 ( xy, u_WindowSize.xy );
 }

// ============================================================================
// == Shader specific inputs ==================================================
// ============================================================================

// The range_x and range_y parameters specify X and Y coordinate ranges that
// define the sides of the primary rectangle.  A secondary rectangle is then
// defined by adding a border either outside or inside the primary rectangle
// depending on the signs of the components of the parameters  border_x  and
// border_y (positive outside, negative inside) specifying the border width.
//
// The the range and border parameters default to effectively  -infinity  to
// +infinity and no border respectively.

uniform vec2 Emu_Rectangle_range_x;  // Range of X for the primary rectangle
uniform vec2 Emu_Rectangle_range_y;  // Range of Y for the primary rectangle

uniform vec2 Emu_Rectangle_border_x; // Widths of the X borders
uniform vec2 Emu_Rectangle_border_y; // Widths of the Y borders

vec2 border_x = EMU_DEFAULT ( vec2, Emu_Rectangle_border_x, vec2(0.0,0.0) );
vec2 border_y = EMU_DEFAULT ( vec2, Emu_Rectangle_border_y, vec2(0.0,0.0) );

vec2 range1_x = EMU_DEFAULT ( vec2, Emu_Rectangle_range_x, vec2(-1.0e30,1.0e30) );
vec2 range1_y = EMU_DEFAULT ( vec2, Emu_Rectangle_range_y, vec2(-1.0e30,1.0e30) );

vec2 range2_x = range1_x + vec2(-1.0,1.0)*border_x; // Range for secondary rectangle
vec2 range2_y = range1_y + vec2(-1.0,1.0)*border_y; // Range for secondary rectangle

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

// The opacity parameter specifies multiplicative factors for the  opacities
// inside and outside the rectangle.  If  there is a border then the opacity
// varies smoothly across it. The default, (1,0),  leaves the opacity of the
// area inside the rectangle unchanged and makes the area outside  be  fully
// transparent.

uniform vec2 Emu_Rectangle_opacity; // Inner and outer opacities

vec2 opacity = EMU_DEFAULT ( vec2, Emu_Rectangle_opacity, vec2(1.0,0.0) );

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

// By default the variation of opacity across the rectangle's border is done
// using the GLSL smoothstep function which has a sigmoidal,  i.e. S shaped,
// graph. This may be overridden via the border_mode control parameter.
//
// Currently the following values for border_mode are recognised
//
//     0 - smoothstep
//     1 - linear
//
// Other border_mode options may be added later.

uniform int Emu_Rectangle_border_mode;

#define border_mode Emu_Rectangle_border_mode

// ============================================================================
// == The shader's major function =============================================
// ============================================================================

vec4 Emu_Rectangle ( vec2 uv )
 {
   // Get the pixel colour from the source image.

   vec4 colour = texture2D ( iChannel0, fract(uv) );

   // Get the inner and outer limits of the boundary.

   vec2 inner_x = vec2 ( max(range1_x[0],range2_x[0]), min(range1_x[1],range2_x[1]) );
   vec2 inner_y = vec2 ( max(range1_y[0],range2_y[0]), min(range1_y[1],range2_y[1]) );

   vec2 outer_x = vec2 ( min(range1_x[0],range2_x[0]), max(range1_x[1],range2_x[1]) );
   vec2 outer_y = vec2 ( min(range1_y[0],range2_y[0]), max(range1_y[1],range2_y[1]) );

   // Calculate the border's fade in and fade out factor.

   float f = 1.0;

   switch ( border_mode ) {

      case 0: // Sigmoidal opacity variation

         f = ( 0.0 + smoothstep ( outer_x[0], inner_x[0], uv.x ) )
           * ( 0.0 + smoothstep ( outer_y[0], inner_y[0], uv.y ) )
           * ( 1.0 - smoothstep ( inner_x[1], outer_x[1], uv.x ) )
           * ( 1.0 - smoothstep ( inner_y[1], outer_y[1], uv.y ) );

           break;

      case 1: // Linear opacity variation

         vec2 bx = vec2 ( (uv.x-outer_x[0])/abs(border_x[0]), 
                          (outer_x[1]-uv.x)/abs(border_x[1]) );

         vec2 by = vec2 ( (uv.y-outer_y[0])/abs(border_y[0]),
                          (outer_y[1]-uv.y)/abs(border_y[1]) );

         bx = clamp ( bx, vec2(0.0), vec2(1.0) );
         by = clamp ( by, vec2(0.0), vec2(1.0) );

         f = bx[0]
           * by[0]
           * bx[1]
           * by[1];

         break;

     }

   // Modify the pixel's opacity.

   colour.a *= mix ( opacity.y, opacity.x, f );

   return colour;

 }

// ============================================================================
// == The shader's main routine ===============================================
// ============================================================================

void main ( void )
 {
   // Get the normalised coordinates of the current point.

   vec2 uv = Emu_Normalise_to_Window ( gl_FragCoord.xy );

   // Apply the rectangle function and update the shader's outputs.

   gl_FragColor = Emu_Rectangle(uv) * gl_Color;

 }

// ============================================================================
